﻿using UnityEngine;
using System.Collections;
using System.Collections.Generic;
using SimplexNoise;
using LibTessDotNet;

//[ExecuteInEditMode]
public class Room : MonoBehaviour
{

    private PolygonCollider2D _collider;
    private int _seed = 52;
    private float _lastEntranceOffset = 9f;
    public int _Level = 0;
    public int _StartLevel = 0;
    public bool _Started = false;
    public bool _GameOver = false;
    public float _GameOverTS;
    public float _GameOverDelay = 1f;
    public GameObject _Exit;
    public Score _Score;
    public Tutorial _Tutorial;
    public GameObject _Ball;
    public GameObject _Island;
    public GameObject _NextRoom;
    public GameObject _NextIsland;

    public Color32 _NextRoomColor = new Color32(200, 200, 200, 128);
    public Color32 _BlockedColor = new Color32(244, 81, 56, 255);
    public Camera _Background;
    public Camera _Main;
    public Texture2D _ScoreScreenshot = null;

    private Vector2[] _currentRoomPoints;
    private Vector2[] _nextRoomPoints;
    private Mover _ballMover;
    private float _lastBlockedTS;
    private AudioSource _audioSource;

    public const float WIDTH = 18f;
    public const float HEIGHT = 32f;
    public const float X_VAR = 1.5f;
    public const float Y_VAR = HEIGHT / 3f;
    public const float TOP_PADDING = HEIGHT / 14f;
    public const float SIDE_PADDING = WIDTH / 7f;
    public const float BOTTOM_PADDING = HEIGHT / 14f;
    public const float EXIT_WIDTH = WIDTH / 5f;
    public const float EXIT_HEIGHT = BOTTOM_PADDING;
    public const float INSIDE_MARGIN = EXIT_WIDTH / 3f;
    public const float OUTSIDE_MARGIN = 32f;

	public bool IsGameOver {
		get {
			return _GameOver;
		}
	}

    // Use this for initialization
    void Start()
    {
        _audioSource = GetComponent<AudioSource>();
        Random.seed = _seed;
        _StartLevel = _Level;
        _collider = GetComponent<PolygonCollider2D>();
        _currentRoomPoints = NewRoom(0);
        //_nextRoomPoints = NewRoom(0);
        NextLevel();
        _ballMover = _Ball.GetComponent<Mover>();
        _NextRoomColor = _NextRoom.GetComponent<Renderer>().material.color;
        _Background.transform.position = new Vector3(WIDTH / 2, HEIGHT / 2, _Background.transform.position.z);
        _Main.transform.position = new Vector3(WIDTH / 2, HEIGHT / 2, _Main.transform.position.z);
        _Exit.transform.localScale = new Vector3(EXIT_WIDTH, EXIT_HEIGHT * 2f, 0f);

    }
	

    // Update is called once per frame
    void LateUpdate()
    {
        if (_NextRoom.GetComponent<Renderer>().material.color != _NextRoomColor && Time.time - _lastBlockedTS > 0.1f)
        {
/*            _NextRoom.renderer.material.color = Color32.Lerp(_NextRoom.renderer.material.color, _NextRoomColor, Time.time - _lastBlockedTS );
            _NextIsland.renderer.material.color = Color32.Lerp(_NextIsland.renderer.material.color, _NextRoomColor, Time.time - _lastBlockedTS);
*/
            _NextRoom.GetComponent<Renderer>().material.color = _NextRoomColor;
            _NextIsland.GetComponent<Renderer>().material.color = _NextRoomColor;
        }

        // UI Blocked for a time after gameover
        if (_GameOver && Time.time < _GameOverTS + _GameOverDelay)
            return;

		// don't process inputs when game is paused (Help screen).
		if (Time.timeScale < 1f) {
			return;
		}

        if (Input.GetKeyDown(KeyCode.Space) && _ballMover._Blocked)
        {
            // block shape update and flash next room red
            _NextRoom.GetComponent<Renderer>().material.color = _BlockedColor;
            _NextIsland.GetComponent<Renderer>().material.color = _BlockedColor;
            _lastBlockedTS = Time.time;
            _NextRoom.GetComponent<AudioSource>().Play();
        }

        if (_Started && Input.GetKeyDown(KeyCode.Space) && !_ballMover._Blocked)
        {
            // shape update
            NextLevel();
            return;
        }

        // start a new game
        if (Input.GetKeyUp(KeyCode.Space) && !_ballMover._Blocked && !_Started)
        {
            NewGame();
            return;
        }

    }

    public void NewGame()
    {
        _Ball.GetComponent<Renderer>().enabled = true;
        _Score.ResetScore();
        _StartLevel = _Level;
        _Started = true;
        _GameOver = false;
        _Exit.SetActive(true);
        _Background.backgroundColor = new Color32(210, 229, 204, 255);
        _NextRoom.gameObject.GetComponent<Renderer>().enabled = true;
        _Island.GetComponent<Renderer>().enabled = true;
        _NextIsland.GetComponent<Renderer>().enabled = true;
        _ballMover.Go();
        _ScoreScreenshot = null;
    }

    public void GameOver(Collider2D coll)
    {
        _Ball.GetComponent<Renderer>().enabled = false;
        _GameOver = true;
        _Started = false;
        _Exit.SetActive(false);
        _GameOverTS = Time.time;
        _Background.backgroundColor = new Color32(244, 81, 56, 255);
        _NextRoom.gameObject.GetComponent<Renderer>().enabled = false;
        _NextIsland.GetComponent<Renderer>().enabled = false;
        coll.GetComponent<Rigidbody2D>().velocity = Vector2.zero;
    }

    void NextLevel()
    {
        _Ball.GetComponent<Renderer>().enabled = true;
        if (_Island.activeSelf)
        {
            _Score.DeductPoints();
            _Island.GetComponent<Island>().Crumble();
        }
        _nextRoomPoints = NewRoom(++_Level);
        _collider.SetPath(0, _currentRoomPoints);

        GetComponent<MeshFilter>().mesh = Triangulator.PolygonMesh(_currentRoomPoints);
        _Exit.transform.position = new Vector3(_currentRoomPoints[4].x, 0, 0);
        _NextRoom.GetComponent<MeshFilter>().mesh = Triangulator.PolygonMesh(_nextRoomPoints);
        _NextRoom.GetComponent<PolygonCollider2D>().points = _nextRoomPoints;
        _currentRoomPoints = _nextRoomPoints;
        _Island.SetActive(true);
        _Island.GetComponent<Island>().Next();
        _Score.ResetIslandScore();
        _audioSource.Play();
    }

    private Vector2[] NewRoom(int offset)
    {
        float topPadding = TOP_PADDING;
        float exitOffset = INSIDE_MARGIN;


        if (_lastEntranceOffset > WIDTH * 0.5f)
        {
            exitOffset = Random.Range(SIDE_PADDING, WIDTH * 0.5f - EXIT_WIDTH);
        }
        else
        {
            exitOffset = Random.Range(WIDTH * 0.5f, WIDTH - SIDE_PADDING - EXIT_WIDTH);
        }
        _lastEntranceOffset = exitOffset;
        List<Vector2> points = new List<Vector2>();

        // define outside shape
        points.Add(new Vector2(-OUTSIDE_MARGIN, 0)); // low-left corner
        points.Add(new Vector2(-OUTSIDE_MARGIN, HEIGHT)); // top-left corner
        points.Add(new Vector2(WIDTH + OUTSIDE_MARGIN, HEIGHT)); // top-right corner
        points.Add(new Vector2(WIDTH + OUTSIDE_MARGIN, 0)); // low-right corner
        points.Add(new Vector2(exitOffset + EXIT_WIDTH * 0.5f, 0)); // exit tip
        points.Add(new Vector2(exitOffset + EXIT_WIDTH, EXIT_HEIGHT)); // left-exit inside

        var lastRef = new Vector2(X_VAR * 0.5f, 0);
        var lastPos = new Vector2(exitOffset + EXIT_WIDTH, EXIT_HEIGHT);
        int leg = 0;
        int legStep = 0;

        // generarte a random zig-zag line that grows in y-axis while
        // staying within certain bounds in x-axis.
        for (int i = 0; i < 64; i++)
        {
            var xMin = lastRef.x * -1; // 
            var xMax = X_VAR + lastRef.x;
            var yMin = lastRef.y + 0.5f; // move forward
            var yMax = lastRef.y + Y_VAR; // but not too much
            var x = Mathf.Clamp(Random.Range(xMin, xMax), -X_VAR, X_VAR);
            var y = Random.Range(yMin, yMax);
            var newRef = new Vector2(x, y);

            if (legStep == 0)
                newRef.y = Mathf.Clamp(newRef.y, INSIDE_MARGIN, yMax);

            var diff = new Vector2(newRef.x - lastRef.x, newRef.y - lastRef.y);
            lastRef = newRef;

            Vector2 newPos;

            // matrix transform of the reference shape to bend it at four
            // corners to form a rectangular shape

            if (leg == 0)
            {
                // right-exit to low-right corner

                newPos = new Vector2(
                    Mathf.Clamp(lastPos.x + diff.y, INSIDE_MARGIN, WIDTH - INSIDE_MARGIN), 
                    Mathf.Clamp(lastPos.y + diff.x, INSIDE_MARGIN, exitOffset + EXIT_WIDTH)
                );

                if (newPos.x > WIDTH - SIDE_PADDING)
                {
                    leg++;
                    legStep = 0;
                }
            }
            else if (leg == 1)
            {
                // right wall
                newPos = new Vector2(
                    Mathf.Clamp(lastPos.x + diff.x, INSIDE_MARGIN, WIDTH - INSIDE_MARGIN), 
                    Mathf.Clamp(lastPos.y + diff.y, INSIDE_MARGIN, HEIGHT - INSIDE_MARGIN)
                );
                if (newPos.y > HEIGHT - topPadding)
                {
                    leg++;
                    legStep = 0;
                }
            }
            else if (leg == 2)
            {
                // top wall
                newPos = new Vector2(
                    Mathf.Clamp(lastPos.x - diff.y, INSIDE_MARGIN, WIDTH - INSIDE_MARGIN), 
                    Mathf.Clamp(lastPos.y - diff.x, INSIDE_MARGIN, HEIGHT - INSIDE_MARGIN)
                );
                if (newPos.x < SIDE_PADDING)
                {
                    leg++;
                    legStep = 0;
                }
            }
            else if (leg == 3)
            {
                // left wall
                newPos = new Vector2(
                    Mathf.Clamp(lastPos.x - diff.x, INSIDE_MARGIN, WIDTH - INSIDE_MARGIN), 
                    Mathf.Clamp(lastPos.y - diff.y, INSIDE_MARGIN, HEIGHT - INSIDE_MARGIN)
                );
                if (newPos.y < BOTTOM_PADDING)
                {
                    leg++;
                    legStep = 0;
                }
            }
            else
            {
                // low-left corner to exit-left inside
                newPos = new Vector2(
                    Mathf.Clamp(lastPos.x + diff.y, exitOffset, WIDTH - INSIDE_MARGIN), 
                    Mathf.Clamp(lastPos.y + diff.x, EXIT_HEIGHT, HEIGHT - INSIDE_MARGIN)
                );
                if (newPos.x > exitOffset - INSIDE_MARGIN)
                    break;
            }

            // prevent shapes that overlap the exit
            if (
                newPos.x < exitOffset &&
                newPos.x > exitOffset * EXIT_WIDTH && 
                newPos.y < EXIT_HEIGHT * 2f &&
                newPos.y > 0)
            {
                newPos.y = EXIT_HEIGHT * 2f;
                if (newPos.x > exitOffset + EXIT_WIDTH * 0.5f)
                    newPos.x = exitOffset + EXIT_WIDTH;
                else
                    newPos.x = exitOffset;
            }

            points.Add(newPos);
            lastPos = newPos;
            legStep++;
        }

        // close the shape at the exit tip
        points.Add(new Vector2(exitOffset, EXIT_HEIGHT));
        points.Add(new Vector2(exitOffset + EXIT_WIDTH * 0.5f, 0));
        return points.ToArray();
    }
}
